//=============================================================================
//   This file is part of VTKEdge. See vtkedge.org for more information.
//
//   Copyright (c) 2010 Kitware, Inc.
//
//   VTKEdge may be used under the terms of the BSD License
//   Please see the file Copyright.txt in the root directory of
//   VTKEdge for further information.
//
//   Alternatively, you may see: 
//
//   http://www.vtkedge.org/vtkedge/project/license.html
//
//
//   For custom extensions, consulting services, or training for
//   this or any other Kitware supported open source project, please
//   contact Kitware at sales@kitware.com.
//
//
//=============================================================================

// .NAME vtkKWEFunctionToGLSL - Parse a mathematical expression and generate a
// GLSL source code string.
// .SECTION Description
// vtkKWEFunctionToGLSL parses a mathematical expression conformed to the
// syntax parsed by vtkFunctionParser. However, instead of evaluating a value by
// executing the bytecode, it uses the bytecode to generate a GLSL source code
// used by vtkKWEGPUArrayCalculator.
//
// The functions that this array calculator understands is:
// <pre>
// standard operations: + - * / ^ .
// access vector components: iHat, jHat, kHat
// abs
// acos
// asin
// atan
// ceil
// cos
// cosh
// exp
// floor
// log
// mag
// min
// max
// norm
// sign
// sin
// sinh
// sqrt
// tan
// tanh
// </pre>
// Note that some of these operations work on scalars, some on vectors, and some on
// both (e.g., you can multiply a scalar times a vector). The operations are performed
// tuple-wise (i.e., tuple-by-tuple). The user must specify which arrays to use as
// vectors and/or scalars, and the name of the output data array.
//
// .SECTION See Also
// vtkFunctionParser vtkKWEGPUArrayCalculator.

#ifndef __vtkKWEFunctionToGLSL_h
#define __vtkKWEFunctionToGLSL_h

#include "vtkFunctionParser.h"
#include "VTKEdgeConfigure.h" // include configuration header

class vtkStdString;
class vtkOStrStreamWrapper;

class VTKEdge_COMMON_EXPORT vtkKWEFunctionToGLSL : public vtkFunctionParser
{
public:
  vtkTypeRevisionMacro(vtkKWEFunctionToGLSL,vtkFunctionParser);
  void PrintSelf(ostream& os, vtkIndent indent);

  static vtkKWEFunctionToGLSL *New();

  // Description:
  // Value modified by GenerateGLSL().
  virtual bool GetParseStatus();

  // Description:
  // Generate GLSL source code in GLSLCode.
  // Update ParseStatus.
  virtual void GenerateGLSL();

  // Description:
  // Return the dimension of the result of the expression.
  // \pre valid_expression: GetParseStatus()
  // \post valid_result: result==1 || result==3
  virtual int GetResultDimension();

  // Description:
  // Return the GLSLCode generated by GenerateGLSL() in a string.
  // \pre valid_expression: GetParseStatus()
  virtual vtkStdString *GetGLSLCode();

  // Description:
  // Return if a given scalar data array is used in the function expression.
  // \pre valid_expression: GetParseStatus()
  // \pre valid_index:index>=0 && index<this->NumberOfScalarVariables
  virtual bool GetScalarIsUsed(int index);

  // Description:
  // Return if a given vector data array is used in the function expression.
  // \pre valid_expression: GetParseStatus()
  // \pre valid_index:index>=0 && index<this->NumberOfVectorVariables
  virtual bool GetVectorIsUsed(int index);

  // Description:
  // Return the GLSL name attached to the a given scalar data array.
  // \pre valid_expression: GetParseStatus()
  // \pre valid_index:index>=0 && index<this->NumberOfScalarVariables
  // \pre used_scalar: this->GetScalarIsUsed(index)
  vtkStdString *GetGLSLScalarName(int index);

  // Description:
  // Return the GLSL name attached to the a given vector data array.
  // \pre valid_expression: GetParseStatus()
  // \pre valid_index:index>=0 && index<this->NumberOfVectorVariables
  // \pre used_vector: this->GetVectorIsUsed(index)
  vtkStdString *GetGLSLVectorName(int index);

  // Description:
  // Return the number of scalar variables used in the expression.
  // \pre valid_expression: GetParseStatus()
  // \post valid_result: result>=0 && result<=GetNumberOfScalarVariables()
  virtual int GetNumberOfUsedScalarVariables();

  // Description:
  // Return the number of vector variables used in the expression.
  // \pre valid_expression: GetParseStatus()
  // \post valid_result: result>=0 && result<=GetNumberOfVectorVariables()
  virtual int GetNumberOfUsedVectorVariables();

protected:
  // Description:
  // Default constructor. The function expression is a null pointer.
  // The GLSLCode is a null pointer. ParseStatus is false.
  vtkKWEFunctionToGLSL();

  // Description:
  // Destructor.
  ~vtkKWEFunctionToGLSL();

  // Description:
  // Allocate string stack.
  // \pre void: this->StringStack==0
  // \pre positive_size: this->StackSize>0
  // \post allocated: this->StringStack!=0
  void AllocateStringStack();

  // Description:
  // Delete string stack.
  void DeleteStringStack();

  // Description:
  // Build the name of the variable for GLSL. Only load the variables used
  // in the function expression.
  // \pre parsed: bytcode exists and is valid
  void BuildGLSLVariableNames();

  bool ParseStatus;
  vtkStdString *GLSLCode;

  vtkStdString *StringStack;
  int *PrecedenceStack;

  bool *ScalarIsUsed;
  int ScalarIsUsedSize;
  vtkStdString *GLSLScalarNames;
  int NumberOfUsedScalarVariables;

  bool *VectorIsUsed;
  int VectorIsUsedSize;
  vtkStdString *GLSLVectorNames;
  int NumberOfUsedVectorVariables;

  int ResultDimension; // 1 or 3

private:
  vtkKWEFunctionToGLSL(const vtkKWEFunctionToGLSL&);  // Not implemented.
  void operator=(const vtkKWEFunctionToGLSL&);  // Not implemented.
};

#endif
